#!/bin/bash

# Check if URL is provided
if [ -z "$1" ]; then
  echo "Usage: $0 <website_url>"
  exit 1
fi

URL=$1
OUTPUT_DIR="output_with_ai"

# Create output directories if they don't exist
mkdir -p "$OUTPUT_DIR"
mkdir -p "$OUTPUT_DIR/screenshots"

echo "Analyzing website: $URL"

# Function to run the demo/simulation mode
run_demo_mode() {
  echo "Running in demo mode..."
  
  # Simulate steps with progress indicators
  echo "Step 1/5: Initializing analysis..."
  sleep 2
  
  echo "Step 2/5: Taking screenshot of $URL..."
  sleep 3
  
  echo "Step 3/5: Analyzing page structure..."
  sleep 2
  
  echo "Step 4/5: Detecting colors and typography..."
  sleep 2
  
  echo "Step 5/5: Generating Tailwind rebuild context..."
  sleep 2
  
  echo "Analysis complete!"
  
  # Prepare the domain
  DOMAIN=$(echo "$URL" | sed -E 's|^https?://||' | sed -E 's|/.*$||')
  TIMESTAMP=$(date -Iseconds)
  SCREENSHOT_NAME="${DOMAIN//[.\/]/_}_$(date +%Y%m%d_%H%M%S).png"
  SCREENSHOT_PATH="$OUTPUT_DIR/screenshots/$SCREENSHOT_NAME"
  
  # Generate a sample rebuild_context.json
  echo "Generating sample rebuild context..."
  cat > "$OUTPUT_DIR/rebuild_context.json" << EOL
{
  "site_info": {
    "title": "Demo Site - $URL",
    "domain": "$DOMAIN",
    "url": "$URL",
    "analyzed_at": "$TIMESTAMP"
  },
  "layout": {
    "sections": [
      {
        "type": "header",
        "content_preview": "Navigation and branding area",
        "tailwind_notes": "Use flex layout with justify-between for logo and nav",
        "ai_comment": {
          "location": "Top of the page, spanning full width",
          "style": "Clean design with subtle background and balanced spacing",
          "implementation": "bg-white shadow-sm py-4 px-6 flex justify-between items-center"
        },
        "logo": {
          "position": "left",
          "dimensions": {
            "width": 120,
            "height": 40
          }
        }
      },
      {
        "type": "hero",
        "content_preview": "Main banner with call to action",
        "tailwind_notes": "Consider grid or flex for responsive layout",
        "ai_comment": {
          "location": "Below header, full width section",
          "style": "Bold typography with accent color for CTA button",
          "implementation": "bg-gray-100 py-16 px-8 md:py-24 md:px-16 text-center md:text-left"
        }
      }
    ]
  },
  "design": {
    "colors": {
      "background": ["#ffffff", "#f8f9fa", "#e9ecef"],
      "text": ["#212529", "#495057", "#6c757d"],
      "accent": ["#007bff", "#0069d9"],
      "suggested_tailwind": {
        "primary": "blue-600",
        "secondary": "gray-600",
        "background": "gray-100",
        "text": "gray-800"
      }
    },
    "typography": {
      "fonts": ["Inter", "system-ui", "sans-serif"],
      "headings": {
        "h1": {
          "font-size": "2.5rem",
          "font-weight": "700",
          "line-height": "1.2"
        },
        "h2": {
          "font-size": "2rem",
          "font-weight": "600",
          "line-height": "1.25"
        }
      }
    }
  },
  "tailwind_rebuild_notes": {
    "general": "Use a mobile-first approach with responsive utilities",
    "layout": "Container with max-width and centered content",
    "components": {
      "buttons": "Consistent padding and rounded corners",
      "cards": "Shadow utilities with hover effects"
    },
    "responsive_breakpoints": {
      "sm": "640px",
      "md": "768px",
      "lg": "1024px",
      "xl": "1280px"
    }
  },
  "ai_verification": {
    "verified_at": "$TIMESTAMP",
    "sections_analysis": "The site has a clean header section with good spacing and a prominent hero section with clear call-to-action elements.",
    "layout_assessment": "Overall layout follows good responsive design practices with appropriate spacing between sections.",
    "logo_and_branding": "Logo is positioned in the top-left with adequate padding and good visibility.",
    "color_analysis": "The color scheme is cohesive with blue accent colors against neutral backgrounds.",
    "responsive_strategy": "Mobile-first approach suggested with breakpoints at standard Tailwind values.",
    "tailwind_recommendations": "Utilize flex and grid layouts with consistent spacing utilities. Consider extracting common patterns into Tailwind components."
  }
}
EOL

  # Create a screenshot with the URL and timestamp
  echo "Creating screenshot..."
  if command -v convert >/dev/null 2>&1; then
    convert -size 1200x800 gradient:white-lightblue -font Arial -pointsize 30 -gravity center -annotate 0 "Demo Screenshot\n$URL\n\nAnalyzed at: $(date)" "$SCREENSHOT_PATH"
    echo "Screenshot created using ImageMagick"
  else
    echo "<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"1200\" height=\"800\">
    <linearGradient id=\"grad\" x1=\"0%\" y1=\"0%\" x2=\"100%\" y2=\"100%\">
      <stop offset=\"0%\" style=\"stop-color:white;stop-opacity:1\" />
      <stop offset=\"100%\" style=\"stop-color:lightblue;stop-opacity:1\" />
    </linearGradient>
    <rect width=\"1200\" height=\"800\" fill=\"url(#grad)\" />
    <text x=\"600\" y=\"400\" font-family=\"Arial\" font-size=\"30\" text-anchor=\"middle\" fill=\"black\">Demo Screenshot</text>
    <text x=\"600\" y=\"450\" font-family=\"Arial\" font-size=\"20\" text-anchor=\"middle\" fill=\"black\">$URL</text>
    <text x=\"600\" y=\"500\" font-family=\"Arial\" font-size=\"16\" text-anchor=\"middle\" fill=\"black\">Analyzed at: $(date)</text>
  </svg>" > "$SCREENSHOT_PATH.svg"
    
    # Try to convert SVG to PNG if possible
    if command -v rsvg-convert >/dev/null 2>&1; then
      rsvg-convert -o "$SCREENSHOT_PATH" "$SCREENSHOT_PATH.svg"
      rm "$SCREENSHOT_PATH.svg"
      echo "Screenshot created using rsvg-convert"
    elif command -v convert >/dev/null 2>&1; then
      convert "$SCREENSHOT_PATH.svg" "$SCREENSHOT_PATH"
      rm "$SCREENSHOT_PATH.svg"
      echo "Screenshot created using convert from SVG"
    else
      # Rename SVG as our screenshot if no converter available
      mv "$SCREENSHOT_PATH.svg" "$SCREENSHOT_PATH"
      echo "Screenshot created as SVG (no PNG converters available)"
    fi
  fi
}

# Execute the main analysis script
# If it exists, run it with the given URL
if [ -f "./main.py" ]; then
  echo "Found main.py script, attempting to execute..."
  
  # Try to run the main script but fall back to demo mode if it fails
  if [ -n "$SKIP_AI" ]; then
    echo "AI verification disabled"
    if ! python3 ./main.py "$URL" -o "$OUTPUT_DIR"; then
      echo "Main script failed due to error, falling back to demo mode"
      run_demo_mode
    fi
  else
    echo "AI verification enabled"
    if ! python3 ./main.py "$URL" -o "$OUTPUT_DIR" --verify; then
      echo "Main script with AI verification failed, falling back to demo mode"
      run_demo_mode
    fi
  fi
else
  # If main.py doesn't exist, just run the demo mode
  echo "Main script not found, running in demo mode"
  run_demo_mode
fi

echo "Analysis complete!"
echo "Results saved to $OUTPUT_DIR/"
echo "To view the full results, open $OUTPUT_DIR/rebuild_context.json"

# Make sure the script exits successfully
exit 0 