#!/usr/bin/env python3
"""
Test OpenAI Detailed ASR Generation
"""

import os
import sys
sys.path.append('lib')

from openai_analyzer import OpenAILayoutAnalyzer
import json

# Sample ASR response to test parsing
sample_asr_response = """
Annotated Structure Reference

Template Structure:

- Header
  Type: Transparent, Absolute Positioning
  Overlay Behavior: Overlaps the hero section
  Contents:
    Company logo (top-left)
    Social icons (email, Twitter, LinkedIn, YouTube)
    Navigation menu (Home, About, ESG, Investors, Gallery, Contact)
    Stock ticker and contact number (top-right)

- Hero Section
  Type: Full Height
  Background: Dark oilfield-themed background image
  Content:
    Overlay text: "Celebrating 20 Years Advancing Canada's Energy Industry"
    Large title: "Enterprise Group Inc"
    Description text about the company
    Buttons: "ASGM Meeting" and "Presentation"

- Navigation
  Location: Inside Header
  Style: Horizontally aligned, light font over dark background
  Sticky/Behavior: Not confirmed from screenshot alone

Main Content

- Section 1: Company Spotlight
  Title: "COMPANY SPOTLIGHT"
  Layout: Two-column layout
  Content:
    Subheading: "Highlighting our commitment to innovation and excellence..."
    Feature: Evolution Power Projects
    Text content with ESG emphasis and technical capabilities
    Right-side product image
    Supporting image gallery
    Buttons: "Learn More" and "FLARE GAS CASE STUDY"

- Section 2: Features/Grid
  Title: "OUR COMPANIES"
  Format: Grid of 4 subsidiary cards
  Elements:
    Companies: Evolution Power Projects, Hart Oilfield Rentals, Arctic Therm, Westar
    Each Card: Logo, preview image, and buttons ("Learn More", "View Details")

Footer
  Content Blocks:
    Service Area Highlight: "Serving Western Canada" with map and CTA buttons
    Quick Links: Navigation to site sections
    Contact Info: Address, phone number, email
    Newsletter Signup: Email input + subscribe button
    Footer Branding: Company mission summary and social icons

Assets
  Logo: Appears top-left and bottom-left
  Image Assets: Multiple product and facility photos
  Videos: Embedded corporate overview and tech showcases
  Buttons/CTAs: Throughout sections for deeper navigation
"""

def test_parsing():
    """Test the ASR parsing with sample data"""
    print("Testing OpenAI detailed ASR parsing...")
    
    try:
        analyzer = OpenAILayoutAnalyzer()
        
        # Test the parsing function directly
        asr_data = analyzer._parse_asr_response(sample_asr_response, "https://example.com")
        
        print("\n=== PARSED ASR DATA ===")
        print(f"Template Structure ({len(asr_data['template_structure'])} sections):")
        for i, structure in enumerate(asr_data['template_structure']):
            print(f"  {i+1}. {structure}")
        
        print(f"\nDetailed Sections ({len(asr_data['detailed_sections'])} sections):")
        for section in asr_data['detailed_sections']:
            print(f"\n--- {section['name']} ---")
            print(f"Type: {section['type']}")
            print(f"Details: {section['details']}")
            if section['content_items']:
                print(f"Content Items: {len(section['content_items'])} items")
                for item in section['content_items'][:3]:  # Show first 3
                    print(f"  - {item}")
                if len(section['content_items']) > 3:
                    print(f"  ... and {len(section['content_items']) - 3} more")
        
        print(f"\nLayout Analysis:")
        layout = asr_data['layout_analysis']
        print(f"  - Layout Type: {layout.get('layout_type', 'unknown')}")
        print(f"  - Total Sections: {layout.get('total_sections', 0)}")
        print(f"  - Has Overlay Header: {layout.get('has_overlay_header', False)}")
        print(f"  - Navigation Style: {layout.get('navigation_style', 'unknown')}")
        
        # Save test results
        with open('test_asr_results.json', 'w') as f:
            json.dump(asr_data, f, indent=2)
        
        print(f"\n✅ Test completed! Results saved to test_asr_results.json")
        return True
        
    except Exception as e:
        print(f"❌ Test failed: {e}")
        import traceback
        traceback.print_exc()
        return False

if __name__ == "__main__":
    test_parsing() 